#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QTimer>
#include <QMessageBox>

QT_USE_NAMESPACE

#define TRUE        1
#define FALSE       0

#define END         0xC0
#define ESC         0xDB
#define ESC_END     0xDC
#define ESC_ESC     0xDD

#define UNUSED(x) (void)(x)

unsigned char InBuff[32];
unsigned char OutBuff[33];

unsigned char SlipTxBuff[128];
unsigned char SlipRxBuff[128];

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    //setWindowFlags(Qt::Widget | Qt::MSWindowsFixedSizeDialogHint);
    this->setFixedWidth(404);
    this->setFixedHeight(440);

    m_bUpdateInputs = true;

    const auto infos = QSerialPortInfo::availablePorts();
    for (const QSerialPortInfo &info : infos)
    {
        QString s = info.portName();

        /*QString s = QObject::tr("Port: ") + info.portName() + "\n"
            + QObject::tr("Location: ") + info.systemLocation() + "\n"
            + QObject::tr("Description: ") + info.description() + "\n"
            + QObject::tr("Manufacturer: ") + info.manufacturer() + "\n"
            + QObject::tr("Serial number: ") + info.serialNumber() + "\n"
            + QObject::tr("Vendor Identifier: ") + (info.hasVendorIdentifier() ? QString::number(info.vendorIdentifier(), 16) : QString()) + "\n"
            + QObject::tr("Product Identifier: ") + (info.hasProductIdentifier() ? QString::number(info.productIdentifier(), 16) : QString()) + "\n"
            + QObject::tr("Busy: ") + (info.isBusy() ? QObject::tr("Yes") : QObject::tr("No")) + "\n";*/

        ui->comboBoxSerialPort->addItem(s);
    }

    m_TableHeader<<"Byte"<<"Input"<<"Output";
    ui->tableWidget->setHorizontalHeaderLabels(m_TableHeader);

    ui->tableWidget->setSelectionBehavior(QAbstractItemView::SelectItems);
    ui->tableWidget->setSelectionMode(QAbstractItemView::SingleSelection);
    //ui->tableWidget->setStyleSheet("QTableView {selection-background-color: red;}");
    //ui->tableWidget->setGeometry(QApplication::desktop()->screenGeometry());

    QTableWidgetItem *item;

    for (int i=0; i<32; i++)
    {
        ui->tableWidget->setRowHeight(i, 18);

        item = new QTableWidgetItem("Byte " + QString::number(i, 10));
        item->setFlags(item->flags() ^ Qt::ItemIsEditable);
        item->setFlags(item->flags() ^ Qt::ItemIsSelectable);
        ui->tableWidget->setItem(i, 0, item);

        ui->tableWidget->setItem(i, 1, new QTableWidgetItem("00"));
        ui->tableWidget->item(i,1)->setBackgroundColor(Qt::green);

        item = new QTableWidgetItem("00");
        item->setFlags(item->flags() ^ Qt::ItemIsEditable);
        item->setFlags(item->flags() ^ Qt::ItemIsSelectable);
        ui->tableWidget->setItem(i, 2, item);
        ui->tableWidget->item(i,2)->setBackgroundColor(Qt::yellow);
    }

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(on_timer()));

    connect(&serialPort, static_cast<void (QSerialPort::*)(QSerialPort::SerialPortError)>(&QSerialPort::error),
              this, &MainWindow::handleError);

    //connect(&serialPort, &QSerialPort::readyRead, this, &MainWindow::readData);

    connect( ui->tableWidget, SIGNAL( cellDoubleClicked (int, int) ), this, SLOT( cellSelected( int, int ) ) );
    connect( ui->tableWidget, SIGNAL( cellChanged (int, int) ), this, SLOT( cellChanged( int, int ) ) );
}

MainWindow::~MainWindow()
{
    timer->stop();

    if (serialPort.isOpen())
    {
        serialPort.close();
    }

    delete timer;

    delete ui;
}

void MainWindow::on_pushButtonStart_clicked()
{
    if (ui->comboBoxSerialPort->currentText() != "")
    {
        QString serialPortName = ui->comboBoxSerialPort->currentText();
        serialPort.setPortName(serialPortName);

        serialPort.setBaudRate(QSerialPort::Baud115200);
        serialPort.setDataBits(QSerialPort::Data8);
        serialPort.setParity(QSerialPort::NoParity);
        serialPort.setStopBits(QSerialPort::OneStop);
        serialPort.setFlowControl(QSerialPort::NoFlowControl);

        if (!serialPort.open(QIODevice::ReadWrite))
        {
            QMessageBox::critical(this, tr("Critical Error"), serialPort.errorString());
            //standardOutput << QObject::tr("Failed to open port %1, error: %2").arg(serialPortName).arg(serialPort.error()) << endl;
            return;
        }

        serialPort.setRequestToSend(false);

        timer->start(100);
    }
}

void MainWindow::on_pushButtonStop_clicked()
{
    timer->stop();

    if (serialPort.isOpen())
    {
        serialPort.close();

        ui->label_USB_Status->setStyleSheet("QLabel { color : blue; }");
        ui->label_EtherCAT_Status->setStyleSheet("QLabel { color : blue; }");

        ui->label_USB_Status->setText("Closed");
        ui->label_EtherCAT_Status->setText("Not Operational");
    }
}

void MainWindow::on_timer()
{
    if (serialPort.isOpen())
    {
        TxSlipFrame();                                              //

        readData();

        InBuff[30] -= 1;                                            // generazione di due denti di sega
        InBuff[31] += 1;                                            // negli ultimi byte per test
    }
}

void MainWindow::readData()
{
    QByteArray readData;

    if (serialPort.waitForReadyRead(100))
    {
        readData = serialPort.readAll();

        if (readData.length() < 36)
        {
            while (serialPort.waitForReadyRead(30))
                readData += serialPort.readAll();
        }
    }
    else
    {
        // Timeout
    }

    int NumChar = readData.length();                          // lettura dati da linea seriale

    if (NumChar < 36)                                           // se non si sono ricevuti almeno
    {                                                           // 36 caratteri il frame non e' valido
        WatchDog();

        if (ui->label_USB_Status->text() != "Error RX")
        {
            ui->label_USB_Status->setStyleSheet("QLabel { color : red; }");
            ui->label_USB_Status->setText("Error RX");
        }
    }
    else                                                        // perche' il frame sia valido la funzione di unwrap
    {                                                           // deve restituire "True" ed il byte di stato                                                               // deve segnalare "operational" (0x08)
        for (int i=0; i<NumChar && i<128; i++)
            SlipRxBuff[i] = readData[i];

        if ((UnWrapSlipFrame(SlipRxBuff, OutBuff) == TRUE) /*&& (OutBuff[32] == 0x08)*/)
        {
            if (OutBuff[32] == 0x08)
            {
                if (ui->label_EtherCAT_Status->text() != "Operational")
                {
                    ui->label_EtherCAT_Status->setStyleSheet("QLabel { color : green; }");
                    ui->label_EtherCAT_Status->setText("Operational");
                }
            }
            else
            {
                WatchDog();
            }

            if (ui->label_USB_Status->text() != "Running")
            {
                ui->label_USB_Status->setStyleSheet("QLabel { color : green; }");
                ui->label_USB_Status->setText("Running");
            }

            UpdateInputs();
            UpdateOutputs();
        }
        else
        {
            WatchDog();

            if (ui->label_USB_Status->text() != "Error RX 2")
            {
                ui->label_USB_Status->setStyleSheet("QLabel { color : red; }");
                ui->label_USB_Status->setText("Error RX 2");
            }
        }
    }
}

void MainWindow::handleError(QSerialPort::SerialPortError error)
{
    if (error == QSerialPort::ResourceError) {
        QMessageBox::critical(this, tr("Critical Error"), serialPort.errorString());

        timer->stop();

        if (serialPort.isOpen())
        {
            serialPort.close();

            ui->label_USB_Status->setStyleSheet("QLabel { color : red; }");
            ui->label_EtherCAT_Status->setStyleSheet("QLabel { color : red; }");

            ui->label_USB_Status->setText("Closed");
            ui->label_EtherCAT_Status->setText("Not Operational");
        }
    }
}

//******************************************************************************************************
//          gestione comunicazione
//*******************************************************************************************************


//---- Trasmissione di un buffer con l'Uart --------------------------------------------------------------
                                                        //
                                                        // si trasmettono 32 byte incapsulati
                                                        // in un frame Slip
void MainWindow::TxSlipFrame(void)
{
    int i;
    int j;

    j = 0;

    SlipTxBuff[j] = END;
    j++;

    for (i=0; i<32; i++)
    {
        switch (InBuff[i])
        {
            case END:
            SlipTxBuff[j] = ESC;
            j++;

            SlipTxBuff[j] = ESC_END;
            j++;
            break;

            case ESC:
            SlipTxBuff[j] = ESC;
            j++;
            SlipTxBuff[j] = ESC_ESC;
            j++;
            break;

            default:
            SlipTxBuff[j] = InBuff[i];
            j++;
            break;
        }
    }

    SlipTxBuff[j] = END;
    j++;

    serialPort.readAll();

    qint64 bytesWritten = serialPort.write((const char *)SlipTxBuff, j);

    if (bytesWritten == -1) {
        //standardOutput << QObject::tr("Failed to write the data to port %1, error: %2").arg(serialPortName).arg(serialPort.errorString()) << endl;
        return;
    } else if (bytesWritten != j) {
        //standardOutput << QObject::tr("Failed to write all the data to port %1, error: %2").arg(serialPortName).arg(serialPort.errorString()) << endl;
        return;
    } else if (!serialPort.waitForBytesWritten(5000)) {
        //standardOutput << QObject::tr("Operation timed out or an error occurred for port %1, error: %2").arg(serialPortName).arg(serialPort.errorString()) << endl;
        return;
    }
}


//---- estrai i dati da un frame Slip --------------------------------------------------
                                                    //
                                                    // il buffer estratto deve
                                                    // essere lungo 33 byte
                                                    // 32 di dati + 1 di stato

unsigned char MainWindow::UnWrapSlipFrame(unsigned char *BuffIn, unsigned char *BuffOut)
{
    int i;
    unsigned char Result = FALSE;
    unsigned char FrameComplete = FALSE;
    unsigned char PreviousRxChar = 0x00;
    unsigned char RxChar;
    unsigned char Len = 0;


    if(BuffIn[0] != END)                            // verifica che il frame inizi con END
    {                                               //
        Result = FALSE;                             //
        return Result;                              //
    }                                               //
    else                                            //----loop di analisi del buffer di ingresso ----
    {
        for (i=1; i<128; i++)
        {
            RxChar = BuffIn[i];

            switch(RxChar)                          // analizza il carattere ricevuto
            {
                case ESC:                           //--- e' ESC: si aspetta il carattere successivo ---
                PreviousRxChar = RxChar;            //
                break;                              //

                case END:                           //--- e' END: il frame e' finito -------------------
                if (Len == 34)                      // se si sono ricevuti 33 caratteri il frame e' valido
                    Result = TRUE;                  //
                else                                //
                    Result = FALSE;                 //
                                                    //
                FrameComplete = TRUE;               // in ogni caso esci
                break;                              //

                default:                            //--- e' un carattere normale ----------------------
                                                    //
                if(PreviousRxChar == ESC)           // se il carattere precedente era ESC bisogna
                {                                   // decodificare il carattere ricevuto
                    PreviousRxChar = RxChar;        //
                                                    //
                    switch(RxChar)                  //
                    {                               //
                        case ESC_END:               // e' un carattere uguale a END
                        RxChar = END;               //
                        break;                      //
                                                    //
                        case ESC_ESC:               // e' un carattere uguale a ESC
                        RxChar = ESC;               //
                        break;                      //
                    }                               //
                }                                   //

                else                                // si usa il carattere senza modificarlo
                {                                   //
                    PreviousRxChar = RxChar;        //
                }                                   //

                BuffOut[Len] = RxChar;              // metti il carattere ricevuto nel buffer
                Len++;                              // incrementa il numero di caratteri ricevuti

                if(Len > 34)                        // se c'e' buffer overflow esci segnalando
                {                                   // frame non valido
                    Result = FALSE;                 //
                    FrameComplete = TRUE;           //
                }                                   //
                break;                              //
            }

            if (FrameComplete == TRUE)              // il frame e' terminato
                break;                              // esci dal loop
        }
    }
    return Result;
}

void MainWindow::WatchDog (void)                                        // gestione watchdog
{
    int i;

    if (ui->label_EtherCAT_Status->text() != "Not Operational")
    {
        ui->label_EtherCAT_Status->setStyleSheet("QLabel { color : red; }");
        ui->label_EtherCAT_Status->setText("Not Operational");
    }

    for (i=0; i<32; i++)                                    // azzera il buffer degli output
    {                                                       //
        OutBuff[i] = 0x00;                                  //
    }                                                       //
}

void MainWindow::UpdateInputs()
{
    if(!m_bUpdateInputs)
        return;

    QString str;

    for (int i=0; i<32; i++)
    {
        QTableWidgetItem * item = ui->tableWidget->item(i, 1);
        str.sprintf("%02X", InBuff[i]);
        item->setText(str);
    }
}

void MainWindow::UpdateOutputs()
{
    QString str;

    for (int i=0; i<32; i++)
    {
        QTableWidgetItem * item = ui->tableWidget->item(i, 2);
        str.sprintf("%02X", OutBuff[i]);
        item->setText(str);
    }
}

void MainWindow::cellSelected(int nRow, int nCol)
{
    UNUSED(nRow);
    UNUSED(nCol);

    if (nCol == 1)
    {
        m_bUpdateInputs = false;
    }
}

void MainWindow::cellChanged(int nRow, int nCol)
{
    if (nCol == 1)
    {
        QTableWidgetItem * item = ui->tableWidget->item(nRow, nCol);

        bool ok;;

        InBuff[nRow] = item->text().toInt(&ok, 16);

        m_bUpdateInputs = true;
    }
}
